import { LeaderboardConstants } from "./Interfaces/LeaderboardConstants";
import { ICameraBadgeControl, ICameraBadgeLeaderboard } from "./Interfaces/LeaderboardRelated";
import { setRenderLayerRecursively, setRenderOrderRecursivelyRelativeToParent } from "./ComponentUtils";

export class CameraBadgeLeaderboard implements ICameraBadgeLeaderboard {
    protected readonly screenTransform = null;
    private simpleLeaderboardControl: ICameraBadgeControl = null;
    private simpleLeaderboard: SceneObject = null;
    private animationTween: Tween<any> = null;

    constructor(scriptComponent: ScriptComponent, singleBadgePrefab: ObjectPrefab, renderOrder: number) {
        this.simpleLeaderboard = singleBadgePrefab.instantiate(scriptComponent.getSceneObject().getParent());
        this.simpleLeaderboard.enabled = true;

        this.simpleLeaderboardControl = this.simpleLeaderboard.getComponent("ScriptComponent") as ICameraBadgeControl;
        this.screenTransform = this.simpleLeaderboardControl.getParentScreenTransform();

        this.screenTransform.anchors.setCenter(LeaderboardConstants.SingleBadge.POSITION_OUT_OF_SCREEN);
        setRenderOrderRecursivelyRelativeToParent(scriptComponent.getSceneObject(), renderOrder);
        setRenderLayerRecursively(this.simpleLeaderboard, scriptComponent.getSceneObject().layer);
    }

    public visualiseEntries(otherRecords: Leaderboard.UserRecord[], currentUserRecord: Leaderboard.UserRecord): void {
        if (!isNull(currentUserRecord)) {
            this.simpleLeaderboardControl.setUserRecord(currentUserRecord);
        }
    }

    public setBitmoji(bitmojiTexture): void {
        this.simpleLeaderboardControl.setBitmoji(bitmojiTexture);
    }

    public show(): void {
        this.animateLeaderboard(true);
    }

    public hide(): void {
        this.animateLeaderboard(false);
    }

    private animateLeaderboard(toShow: boolean): void {
        const startPos = this.screenTransform.anchors.getCenter();
        const endPos = toShow ? LeaderboardConstants.SingleBadge.POSITION : LeaderboardConstants.SingleBadge.POSITION_OUT_OF_SCREEN;
        const duration = startPos.distance(endPos) * LeaderboardConstants.SingleBadge.ANIMATION_DURATION_MS / 2;
        if (duration === 0) {
            return;
        }

        if (!isNull(this.animationTween)) {
            try {
                this.animationTween.stop();
                TWEEN.remove(this.animationTween);
            } catch (e) {}
        }

        this.animationTween = new TWEEN.Tween({ x: startPos.x, y: startPos.y })
            .to({ x: endPos.x, y: endPos.y }, duration)
            .onUpdate((obj) => {
                this.screenTransform.anchors.setCenter(new vec2(obj.x, obj.y));
            })
            .easing(TWEEN.Easing.Quadratic.InOut);
        this.animationTween.start();
    }
}
